##
## window.py -- Display window with associated storage.
##
## Copyleft 1997 Teemu Kalvas <chery@s2.org>
##

## +class Window       (screen,  screen on which to draw
##                      next,    next window initializer
##                      prev,    previous window initializer
##                      name,    name initializer
##                      server)  server initializer
##  -method page_up    ()
##  -method page_down  ()
##  -method set_name   (name)    new window name
##  -method log        (str)     string to log
##  -method redraw     ()
##  -method next
##  -method prev
##  -method name
##  -method server

import curses

import linelist

import config

def wrap(line, width):
    wrapped = []
    while len(line) >= width:
	wrapped.append(line[:width -1 ] + '\\')
	line = line[width - 1:]
    wrapped.append(line)
    return wrapped

def wlen(line, width):
    return (len(line) + width - 2) / (width - 1)

class Window:
    def __init__(self, screen, next, prev, name, server):
	self.screen = screen
	self.next = next
	self.prev = prev
	self.name = name
	self.server = server
	## at is actually the (line, part) of the first line off the bottom of
	## the display area.  Thus it is always greater than or equal to (0, 0)
	## and less than or equal to (lines.lines, 0).
	self.at = 0, 0
	self.lines = linelist.LineList(8)
	self.hold = 0
	self.detect = 0
	screen.touched = screen.touched | 8

    def page_up(self):
	height, width = self.screen.log_area.getmaxyx()
	i = height - 1
	while i:
	    if self.at[1] > 0:
		self.at = self.at[0], self.at[1] - 1
	    elif self.at[0] > 0:
		self.at = (self.at[0] - 1,
			   wlen(self.lines.lookup(self.at[0] - 1), width) - 1)
	    else:
		break
	    if self.at[0] < self.lines.lines:
		self.hold = 1
	    i = i - 1
	self.screen.touched = self.screen.touched | 8

    def page_down(self):
	height, width = self.screen.log_area.getmaxyx()
	i = height - 1
	l = wlen(self.lines.lookup(self.at[0]), width) - self.at[1]
	while i:
	    if self.at[1] < l - 1:
		self.at = self.at[0], self.at[1] + 1
	    elif self.at[0] < self.lines.lines:
		self.at = self.at[0] + 1, 0
		if self.at[0] == self.lines.lines:
		    self.hold = 0
		    break
		l = wlen(self.lines.lookup(self.at[0]), width)
	    else:
		## This should never be reached.
		break
	    i = i - 1
	self.screen.touched = self.screen.touched | 8

    def set_name(self, name):
	self.name = name
	self.screen.touched = self.screen.touched | 16

    def log(self, str):
	visible = self == self.screen.current
	window = self.screen.log_area
	height, width = window.getmaxyx()
	wrapped = wrap(str, width)
	self.lines.append(str)
	if not self.hold:
	    self.at = self.at[0] + 1, 0
	    if visible:
		for line in wrapped:
		    window.scroll()
		    window.insstr(height - 1, 0, line)
		self.screen.touched = self.screen.touched | 1
	if not visible:
	    self.detect = self.detect + 1
	    self.screen.touched = self.screen.touched | 16

    def redraw(self):
	window = self.screen.log_area
	height, width = window.getmaxyx()
	window.clear()
	y, i, j = height - 1, self.at[0], self.at[1]
	if j > 0:
	    wrapped = wrap(self.lines.lookup(i), width)
	while y >= 0:
	    j = j - 1
	    if j < 0:
		i = i - 1
		if i < 0:
		    self.screen.touched = self.screen.touched | 1
		    return
		wrapped = wrap(self.lines.lookup(i), width)
		j = len(wrapped) - 1
	    window.insstr(y, 0, wrapped[j])
	    y = y - 1
	self.screen.touched = self.screen.touched | 1

## End. ##
